---
-- StrawHarvestSellingStation
--
-- Author: Stijn Wopereis (Wopster)
-- Purpose: Fix the vanilla selling station due to lack of completeness (can't differentiate between pallets or whatever tip vehicle).
--
-- Copyright (c) Creative Mesh UG, 2020

---@class StrawHarvestSellingStation
StrawHarvestSellingStation = {}

local StrawHarvestSellingStation_mt = Class(StrawHarvestSellingStation, SellingStation)

InitObjectClass(StrawHarvestSellingStation, "StrawHarvestSellingStation")

function StrawHarvestSellingStation:new(isServer, isClient, customMt)
    local self = SellingStation:new(isServer, isClient, customMt or StrawHarvestSellingStation_mt)

    self.palletsInTrigger = {}
    registerObjectClassName(self, "StrawHarvestSellingStation")

    return self
end

function StrawHarvestSellingStation:load(rootNode, xmlFile, xmlNode, target, ...)
    if not StrawHarvestSellingStation:superClass().load(self, rootNode, xmlFile, xmlNode, target, ...) then
        return false
    end

    local triggerNode = XMLUtil.getValueFromXMLFileOrUserAttribute(xmlFile, xmlNode, "triggerNode", getXMLString, rootNode)
    self.palletLitersPerSecond = Utils.getNoNil(XMLUtil.getValueFromXMLFileOrUserAttribute(xmlFile, xmlNode, "palletLitersPerSecond", getXMLFloat, rootNode), 10)

    if triggerNode ~= nil then
        self.triggerNode = I3DUtil.indexToObject(rootNode, triggerNode)
        if self.triggerNode ~= nil then
            addTrigger(self.triggerNode, "triggerPalletCallback", self)
        end
    end

    --TODO: Remove temp fix when Giants fixed this issue for clients.
    self.lastIncomeName = self.incomeName

    return true
end

function StrawHarvestSellingStation:delete()
    if self.triggerNode ~= nil then
        removeTrigger(self.triggerNode)
        self.triggerNode = nil
    end

    StrawHarvestSellingStation:superClass().delete(self)
end

---Consume pallets in trigger on update frame.
function StrawHarvestSellingStation:update(dt)
    StrawHarvestSellingStation:superClass().update(self, dt)

    if not self.isServer then
        return
    end

    local inTrigger = 0
    for object, fillType in pairs(self.palletsInTrigger) do
        if object ~= nil and fillType ~= nil and not object.isDeleted then
            local spec = object.spec_mountable
            if spec == nil or spec.dynamicMountJointIndex == nil then
                local fillUnitIndex = object:getFirstValidFillUnitToFill(fillType, true)
                local delta = self.palletLitersPerSecond * dt

                if delta > 0 then
                    --Pass through trigger toolType as we load from the trigger.
                    local applied = self:addFillLevelFromTool(object:getOwnerFarmId(), delta, fillType, nil, ToolType.TRIGGER)
                    object:addFillUnitFillLevel(object:getOwnerFarmId(), fillUnitIndex, -applied, fillType, ToolType.UNDEFINED)

                    if object:getFillUnitFillLevelPercentage(fillUnitIndex) == 0 then
                        self.palletsInTrigger[object] = nil
                        break
                    end
                end
            end
        else
            self.palletsInTrigger[object] = nil
        end

        inTrigger = inTrigger + 1
    end

    if inTrigger > 0 then
        self:raiseActive()
    end
end

---Overwrite tool type allowance due to Giants not differentiating between ToolTypes for pallets and any other vehicle.
function StrawHarvestSellingStation:getIsToolTypeAllowed(toolType)
    if not StrawHarvestSellingStation:superClass().getIsToolTypeAllowed(self, toolType) then
        return false
    end

    return toolType == ToolType.BALE or toolType == ToolType.TRIGGER
end

---Detect valid pallets in the trigger.
function StrawHarvestSellingStation:triggerPalletCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    local object = g_currentMission:getNodeObject(otherId)

    if object ~= nil and object.isa ~= nil and object:isa(Vehicle) and object.typeName:find("pallet") then
        if onEnter then
            local fillUnitIndex = object:getFirstValidFillUnitToFill(FillType.STRAWPELLETS, true)
            if fillUnitIndex == nil then
                fillUnitIndex = object:getFirstValidFillUnitToFill(FillType.HAYPELLETS, true)
            end

            local fillType = object:getFillUnitFillType(fillUnitIndex)
            if fillType ~= nil then
                if self:getIsFillTypeAllowed(fillType) and object:getFillUnitFillLevel(fillUnitIndex) > 0 then
                    self:raiseActive()
                    self.palletsInTrigger[object] = fillType
                end
            end
        elseif onLeave then
            self.palletsInTrigger[object] = nil
        end
    end
end
