---
-- StrawHarvestSellingStationPlaceable
--
-- Author: Stijn Wopereis (Wopster)
-- Purpose: Base placeable class for the selling station.
--
-- Copyright (c) Creative Mesh UG, 2020

---@class StrawHarvestSellingStationPlaceable
StrawHarvestSellingStationPlaceable = {}

local StrawHarvestSellingStationPlaceable_mt = Class(StrawHarvestSellingStationPlaceable, Placeable)

InitObjectClass(StrawHarvestSellingStationPlaceable, "StrawHarvestSellingStationPlaceable")

function StrawHarvestSellingStationPlaceable:new(isServer, isClient, mt)
    local self = Placeable:new(isServer, isClient, mt or StrawHarvestSellingStationPlaceable_mt)

    registerObjectClassName(self, "StrawHarvestSellingStationPlaceable")

    return self
end

function StrawHarvestSellingStationPlaceable:delete()
    if self.sellingStation ~= nil then
        g_currentMission.storageSystem:removeUnloadingStation(self.sellingStation)
        self.sellingStation:delete()
    end

    unregisterObjectClassName(self)
    StrawHarvestSellingStationPlaceable:superClass().delete(self)
end

function StrawHarvestSellingStationPlaceable:load(xmlFilename, x, y, z, rx, ry, rz, initRandom)
    if not StrawHarvestSellingStationPlaceable:superClass().load(self, xmlFilename, x, y, z, rx, ry, rz, initRandom) then
        return false
    end

    local xmlFile = loadXMLFile("TempXML", xmlFilename)

    self.sellingStation = StrawHarvestSellingStation:new(self.isServer, self.isClient)
    self.sellingStation:load(self.nodeId, xmlFile, "placeable.sellingStation", self.customEnvironment)
    self.sellingStation.owningPlaceable = self

    delete(xmlFile)

    return true
end

function StrawHarvestSellingStationPlaceable:finalizePlacement()
    StrawHarvestSellingStationPlaceable:superClass().finalizePlacement(self)
    self.sellingStation:register(true)
    g_currentMission.storageSystem:addUnloadingStation(self.sellingStation)
end

function StrawHarvestSellingStationPlaceable:readStream(streamId, connection)
    StrawHarvestSellingStationPlaceable:superClass().readStream(self, streamId, connection)
    if connection:getIsServer() then
        local sellingStationId = NetworkUtil.readNodeObjectId(streamId)
        self.sellingStation:readStream(streamId, connection)
        g_client:finishRegisterObject(self.sellingStation, sellingStationId)
    end
end

function StrawHarvestSellingStationPlaceable:writeStream(streamId, connection)
    StrawHarvestSellingStationPlaceable:superClass().writeStream(self, streamId, connection)
    if not connection:getIsServer() then
        NetworkUtil.writeNodeObjectId(streamId, NetworkUtil.getObjectId(self.sellingStation))
        self.sellingStation:writeStream(streamId, connection)
        g_server:registerObjectInStream(connection, self.sellingStation)
    end
end

function StrawHarvestSellingStationPlaceable:collectPickObjects(node)
    local foundNode = false
    for _, unloadTrigger in ipairs(self.sellingStation.unloadTriggers) do
        if node == unloadTrigger.exactFillRootNode then
            foundNode = true
            break
        end
    end

    if not foundNode then
        StrawHarvestSellingStationPlaceable:superClass().collectPickObjects(self, node)
    end
end

function StrawHarvestSellingStationPlaceable:loadFromXMLFile(xmlFile, key, resetVehicles)
    if not StrawHarvestSellingStationPlaceable:superClass().loadFromXMLFile(self, xmlFile, key, resetVehicles) then
        return false
    end

    if not self.sellingStation:loadFromXMLFile(xmlFile, key .. ".sellingStation") then
        return false
    end

    return true
end

function StrawHarvestSellingStationPlaceable:saveToXMLFile(xmlFile, key, usedModNames)
    StrawHarvestSellingStationPlaceable:superClass().saveToXMLFile(self, xmlFile, key, usedModNames)

    self.sellingStation:saveToXMLFile(xmlFile, key .. ".sellingStation", usedModNames)
end
